/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.io.orc;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hive.ql.io.filters.BloomFilterIO;
import org.apache.hadoop.hive.ql.io.orc.ColumnStatistics;
import org.apache.hadoop.hive.ql.io.orc.ColumnStatisticsImpl;
import org.apache.hadoop.hive.ql.io.orc.CompressionKind;
import org.apache.hadoop.hive.ql.io.orc.Metadata;
import org.apache.hadoop.hive.ql.io.orc.OrcFile;
import org.apache.hadoop.hive.ql.io.orc.OrcProto;
import org.apache.hadoop.hive.ql.io.orc.OrcRawRecordMerger;
import org.apache.hadoop.hive.ql.io.orc.OrcRecordUpdater;
import org.apache.hadoop.hive.ql.io.orc.OrcStruct;
import org.apache.hadoop.hive.ql.io.orc.OrcUnion;
import org.apache.hadoop.hive.ql.io.orc.Reader;
import org.apache.hadoop.hive.ql.io.orc.RecordReader;
import org.apache.hadoop.hive.ql.io.orc.RecordReaderImpl;
import org.apache.hadoop.hive.ql.io.orc.StripeInformation;
import org.apache.hadoop.hive.ql.io.orc.StripeStatistics;
import org.apache.hadoop.hive.ql.io.orc.Writer;
import org.apache.hadoop.hive.serde2.io.ByteWritable;
import org.apache.hadoop.hive.serde2.io.DoubleWritable;
import org.apache.hadoop.hive.serde2.io.ShortWritable;
import org.apache.hadoop.hive.serde2.objectinspector.ObjectInspectorFactory;
import org.apache.hadoop.hive.serde2.objectinspector.StandardStructObjectInspector;
import org.apache.hadoop.io.BooleanWritable;
import org.apache.hadoop.io.FloatWritable;
import org.apache.hadoop.io.IntWritable;
import org.apache.hadoop.io.LongWritable;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONWriter;

public final class FileDump {
    private static final String UNKNOWN = "UNKNOWN";
    public static final String SEPARATOR = Strings.repeat("_", 120) + "\n";
    public static final int DEFAULT_BLOCK_SIZE = 0x10000000;
    public static final String DEFAULT_BACKUP_PATH = System.getProperty("java.io.tmpdir");
    public static final PathFilter HIDDEN_AND_SIDE_FILE_FILTER = new PathFilter(){

        public boolean accept(Path p) {
            String name = p.getName();
            return !name.startsWith("_") && !name.startsWith(".") && !name.endsWith("_flush_length");
        }
    };

    private FileDump() {
    }

    public static void main(String[] args) throws Exception {
        Configuration conf = new Configuration();
        ArrayList<Integer> rowIndexCols = null;
        Options opts = FileDump.createOptions();
        CommandLine cli = new GnuParser().parse(opts, args);
        if (cli.hasOption('h')) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("orcfiledump", opts);
            return;
        }
        boolean dumpData = cli.hasOption('d');
        boolean recover = cli.hasOption("recover");
        boolean skipDump = cli.hasOption("skip-dump");
        String backupPath = DEFAULT_BACKUP_PATH;
        if (cli.hasOption("backup-path")) {
            backupPath = cli.getOptionValue("backup-path");
        }
        if (cli.hasOption("rowindex")) {
            String[] colStrs = cli.getOptionValue("rowindex").split(",");
            rowIndexCols = new ArrayList<Integer>(colStrs.length);
            for (String colStr : colStrs) {
                rowIndexCols.add(Integer.parseInt(colStr));
            }
        }
        boolean printTimeZone = false;
        if (cli.hasOption('t')) {
            printTimeZone = true;
        }
        String[] files = cli.getArgs();
        ArrayList<String> filesInPath = Lists.newArrayList();
        for (String filename : files) {
            Path path = new Path(filename);
            filesInPath.addAll(FileDump.getAllFilesInPath(path, conf));
        }
        if (dumpData) {
            FileDump.printData(filesInPath, conf);
        } else if (recover && skipDump) {
            FileDump.recoverFiles(filesInPath, conf, backupPath);
        } else {
            FileDump.printMetaData(filesInPath, conf, rowIndexCols, printTimeZone, recover, backupPath);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    static Reader getReader(Path path, Configuration conf, List<String> corruptFiles) throws IOException {
        FileSystem fs = path.getFileSystem(conf);
        long dataFileLen = fs.getFileStatus(path).getLen();
        System.err.println("Processing data file " + path + " [length: " + dataFileLen + "]");
        Path sideFile = OrcRecordUpdater.getSideFile(path);
        boolean sideFileExists = fs.exists(sideFile);
        boolean openDataFile = false;
        boolean openSideFile = false;
        if (fs instanceof DistributedFileSystem) {
            DistributedFileSystem dfs = (DistributedFileSystem)fs;
            openDataFile = !dfs.isFileClosed(path);
            boolean bl = openSideFile = sideFileExists && !dfs.isFileClosed(sideFile);
        }
        if (openDataFile || openSideFile) {
            if (openDataFile && openSideFile) {
                System.err.println("Unable to perform file dump as " + path + " and " + sideFile + " are still open for writes.");
                return null;
            } else if (openSideFile) {
                System.err.println("Unable to perform file dump as " + sideFile + " is still open for writes.");
                return null;
            } else {
                System.err.println("Unable to perform file dump as " + path + " is still open for writes.");
            }
            return null;
        }
        Reader reader = null;
        if (!sideFileExists) return OrcFile.createReader(path, OrcFile.readerOptions(conf));
        long maxLen = OrcRawRecordMerger.getLastFlushLength(fs, path);
        long sideFileLen = fs.getFileStatus(sideFile).getLen();
        System.err.println("Found flush length file " + sideFile + " [length: " + sideFileLen + ", maxFooterOffset: " + maxLen + "]");
        if (maxLen == -1L) {
            if (dataFileLen <= maxLen) return null;
            System.err.println("Data file has more data than max footer offset:" + maxLen + ". Adding data file to recovery list.");
            if (corruptFiles == null) return null;
            corruptFiles.add(path.toUri().toString());
            return null;
        }
        try {
            reader = OrcFile.createReader(path, OrcFile.readerOptions(conf).maxLength(maxLen));
            if (dataFileLen <= maxLen) return reader;
            System.err.println("Data file has more data than max footer offset:" + maxLen + ". Adding data file to recovery list.");
            if (corruptFiles == null) return reader;
            corruptFiles.add(path.toUri().toString());
            return reader;
        }
        catch (Exception e) {
            if (corruptFiles != null) {
                corruptFiles.add(path.toUri().toString());
            }
            System.err.println("Unable to read data from max footer offset. Adding data file to recovery list.");
            return null;
        }
    }

    public static Collection<String> getAllFilesInPath(Path path, Configuration conf) throws IOException {
        ArrayList<String> filesInPath = Lists.newArrayList();
        FileSystem fs = path.getFileSystem(conf);
        FileStatus fileStatus = fs.getFileStatus(path);
        if (fileStatus.isDir()) {
            FileStatus[] fileStatuses;
            for (FileStatus fileInPath : fileStatuses = fs.listStatus(path, HIDDEN_AND_SIDE_FILE_FILTER)) {
                if (fileInPath.isDir()) {
                    filesInPath.addAll(FileDump.getAllFilesInPath(fileInPath.getPath(), conf));
                    continue;
                }
                filesInPath.add(fileInPath.getPath().toString());
            }
        } else {
            filesInPath.add(path.toString());
        }
        return filesInPath;
    }

    private static void printMetaData(List<String> files, Configuration conf, List<Integer> rowIndexCols, boolean printTimeZone, boolean recover, String backupPath) throws IOException {
        ArrayList<String> corruptFiles = Lists.newArrayList();
        for (String filename : files) {
            FileDump.printMetaDataImpl(filename, conf, rowIndexCols, printTimeZone, corruptFiles);
            System.out.println(SEPARATOR);
        }
        if (!corruptFiles.isEmpty()) {
            if (recover) {
                FileDump.recoverFiles(corruptFiles, conf, backupPath);
            } else {
                System.err.println(corruptFiles.size() + " file(s) are corrupted." + " Run the following command to recover corrupted files.\n");
                String fileNames = Joiner.on(" ").skipNulls().join(corruptFiles);
                System.err.println("hive --orcfiledump --recover --skip-dump " + fileNames);
                System.out.println(SEPARATOR);
            }
        }
    }

    private static void printMetaDataImpl(String filename, Configuration conf, List<Integer> rowIndexCols, boolean printTimeZone, List<String> corruptFiles) throws IOException {
        int i;
        Path file = new Path(filename);
        Reader reader = FileDump.getReader(file, conf, corruptFiles);
        if (reader == null) {
            return;
        }
        System.out.println("Structure for " + filename);
        System.out.println("File Version: " + reader.getFileVersion().getName() + " with " + (Object)((Object)reader.getWriterVersion()));
        RecordReaderImpl rows = (RecordReaderImpl)reader.rows();
        System.out.println("Rows: " + reader.getNumberOfRows());
        System.out.println("Compression: " + (Object)((Object)reader.getCompression()));
        if (reader.getCompression() != CompressionKind.NONE) {
            System.out.println("Compression size: " + reader.getCompressionSize());
        }
        System.out.println("Type: " + reader.getObjectInspector().getTypeName());
        System.out.println("\nStripe Statistics:");
        Metadata metadata = reader.getMetadata();
        for (int n = 0; n < metadata.getStripeStatistics().size(); ++n) {
            System.out.println("  Stripe " + (n + 1) + ":");
            StripeStatistics ss = metadata.getStripeStatistics().get(n);
            for (i = 0; i < ss.getColumnStatistics().length; ++i) {
                System.out.println("    Column " + i + ": " + ss.getColumnStatistics()[i].toString());
            }
        }
        ColumnStatistics[] stats = reader.getStatistics();
        int colCount = stats.length;
        System.out.println("\nFile Statistics:");
        for (i = 0; i < stats.length; ++i) {
            System.out.println("  Column " + i + ": " + stats[i].toString());
        }
        System.out.println("\nStripes:");
        int stripeIx = -1;
        for (StripeInformation stripe : reader.getStripes()) {
            ++stripeIx;
            long stripeStart = stripe.getOffset();
            OrcProto.StripeFooter footer = rows.readStripeFooter(stripe);
            if (printTimeZone) {
                String tz = footer.getWriterTimezone();
                if (tz == null || tz.isEmpty()) {
                    tz = UNKNOWN;
                }
                System.out.println("  Stripe: " + stripe.toString() + " timezone: " + tz);
            } else {
                System.out.println("  Stripe: " + stripe.toString());
            }
            long sectionStart = stripeStart;
            for (OrcProto.Stream section : footer.getStreamsList()) {
                String kind = section.hasKind() ? section.getKind().name() : UNKNOWN;
                System.out.println("    Stream: column " + section.getColumn() + " section " + kind + " start: " + sectionStart + " length " + section.getLength());
                sectionStart += section.getLength();
            }
            for (int i2 = 0; i2 < footer.getColumnsCount(); ++i2) {
                OrcProto.ColumnEncoding encoding = footer.getColumns(i2);
                StringBuilder buf = new StringBuilder();
                buf.append("    Encoding column ");
                buf.append(i2);
                buf.append(": ");
                buf.append(encoding.getKind());
                if (encoding.getKind() == OrcProto.ColumnEncoding.Kind.DICTIONARY || encoding.getKind() == OrcProto.ColumnEncoding.Kind.DICTIONARY_V2) {
                    buf.append("[");
                    buf.append(encoding.getDictionarySize());
                    buf.append("]");
                }
                System.out.println(buf);
            }
            if (rowIndexCols == null || rowIndexCols.isEmpty()) continue;
            boolean[] sargColumns = new boolean[colCount];
            for (int colIdx : rowIndexCols) {
                sargColumns[colIdx] = true;
            }
            RecordReaderImpl.Index indices = rows.readRowIndex(stripeIx, null, sargColumns);
            for (int col : rowIndexCols) {
                StringBuilder buf = new StringBuilder();
                String rowIdxString = FileDump.getFormattedRowIndices(col, indices.getRowGroupIndex());
                buf.append(rowIdxString);
                String bloomFilString = FileDump.getFormattedBloomFilters(col, indices.getBloomFilterIndex());
                buf.append(bloomFilString);
                System.out.println(buf);
            }
        }
        FileSystem fs = file.getFileSystem(conf);
        long fileLen = fs.getContentSummary(file).getLength();
        long paddedBytes = FileDump.getTotalPaddingSize(reader);
        double percentPadding = (double)paddedBytes / (double)fileLen * 100.0;
        DecimalFormat format = new DecimalFormat("##.##");
        System.out.println("\nFile length: " + fileLen + " bytes");
        System.out.println("Padding length: " + paddedBytes + " bytes");
        System.out.println("Padding ratio: " + format.format(percentPadding) + "%");
        OrcRecordUpdater.AcidStats acidStats = OrcRecordUpdater.parseAcidStats(reader);
        if (acidStats != null) {
            System.out.println("ACID stats:" + acidStats);
        }
        rows.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void recoverFiles(List<String> corruptFiles, Configuration conf, String backup) throws IOException {
        for (String corruptFile : corruptFiles) {
            System.err.println("Recovering file " + corruptFile);
            Path corruptPath = new Path(corruptFile);
            FileSystem fs = corruptPath.getFileSystem(conf);
            FSDataInputStream fdis = fs.open(corruptPath);
            try {
                long corruptFileLen;
                int toRead;
                ArrayList<Long> footerOffsets = Lists.newArrayList();
                for (long remaining = corruptFileLen = fs.getFileStatus(corruptPath).getLen(); remaining > 0L; remaining -= (long)toRead) {
                    toRead = (int)Math.min(0x10000000L, remaining);
                    byte[] data = new byte[toRead];
                    long startPos = corruptFileLen - remaining;
                    fdis.readFully(startPos, data, 0, toRead);
                    int index = 0;
                    while (index != -1) {
                        long nextFooterOffset;
                        index = FileDump.indexOf(data, "ORC".getBytes(), index + 1);
                        if (index == -1 || !FileDump.isReadable(corruptPath, conf, nextFooterOffset = startPos + (long)index + (long)"ORC".length() + 1L)) continue;
                        footerOffsets.add(nextFooterOffset);
                    }
                    System.err.println("Scanning for valid footers - startPos: " + startPos + " toRead: " + toRead + " remaining: " + remaining);
                }
                System.err.println("Readable footerOffsets: " + footerOffsets);
                FileDump.recoverFile(corruptPath, fs, conf, footerOffsets, backup);
            }
            catch (Exception e) {
                Path recoveryFile = FileDump.getRecoveryFile(corruptPath);
                if (fs.exists(recoveryFile)) {
                    fs.delete(recoveryFile, false);
                }
                System.err.println("Unable to recover file " + corruptFile);
                e.printStackTrace();
                System.err.println(SEPARATOR);
                continue;
            }
            finally {
                fdis.close();
                continue;
            }
            System.err.println(corruptFile + " recovered successfully!");
            System.err.println(SEPARATOR);
        }
    }

    private static void recoverFile(Path corruptPath, FileSystem fs, Configuration conf, List<Long> footerOffsets, String backup) throws IOException {
        Path recoveredPath = FileDump.getRecoveryFile(corruptPath);
        if (fs.exists(recoveredPath)) {
            fs.delete(recoveredPath, false);
        }
        if (footerOffsets == null || footerOffsets.isEmpty()) {
            System.err.println("No readable footers found. Creating empty orc file.");
            StandardStructObjectInspector soi = ObjectInspectorFactory.getStandardStructObjectInspector(Lists.newArrayList(), Lists.newArrayList(), Lists.newArrayList());
            Writer writer = OrcFile.createWriter(recoveredPath, OrcFile.writerOptions(conf).inspector(soi));
            writer.close();
        } else {
            FSDataInputStream fdis = fs.open(corruptPath);
            FileStatus fileStatus = fs.getFileStatus(corruptPath);
            FSDataOutputStream fdos = fs.create(recoveredPath, true, conf.getInt("io.file.buffer.size", 4096), fileStatus.getReplication(), fileStatus.getBlockSize());
            try {
                long fileLen;
                int toRead;
                for (long remaining = fileLen = footerOffsets.get(footerOffsets.size() - 1).longValue(); remaining > 0L; remaining -= (long)toRead) {
                    toRead = (int)Math.min(0x10000000L, remaining);
                    byte[] data = new byte[toRead];
                    long startPos = fileLen - remaining;
                    fdis.readFully(startPos, data, 0, toRead);
                    fdos.write(data);
                    System.err.println("Copying data to recovery file - startPos: " + startPos + " toRead: " + toRead + " remaining: " + remaining);
                }
            }
            catch (Exception e) {
                fs.delete(recoveredPath, false);
                throw new IOException(e);
            }
            finally {
                fdis.close();
                fdos.close();
            }
        }
        if (FileDump.isReadable(recoveredPath, conf, Long.MAX_VALUE)) {
            String scheme = corruptPath.toUri().getScheme();
            String authority = corruptPath.toUri().getAuthority();
            String filePath = corruptPath.toUri().getPath();
            Path backupDataPath = backup.equals(DEFAULT_BACKUP_PATH) ? new Path(scheme, authority, DEFAULT_BACKUP_PATH + filePath) : Path.mergePaths((Path)new Path(backup), (Path)corruptPath);
            FileDump.moveFiles(fs, corruptPath, backupDataPath);
            Path sideFilePath = OrcRecordUpdater.getSideFile(corruptPath);
            Path backupSideFilePath = new Path(backupDataPath.getParent(), sideFilePath.getName());
            FileDump.moveFiles(fs, sideFilePath, backupSideFilePath);
            FileDump.moveFiles(fs, recoveredPath, corruptPath);
            System.err.println("Validation of recovered file successful!");
        }
    }

    private static void moveFiles(FileSystem fs, Path src, Path dest) throws IOException {
        try {
            if (!fs.exists(dest.getParent())) {
                fs.mkdirs(dest.getParent());
            }
            fs.delete(dest, false);
            if (!fs.rename(src, dest)) {
                throw new IOException("Unable to move " + src + " to " + dest);
            }
            System.err.println("Moved " + src + " to " + dest);
        }
        catch (Exception e) {
            throw new IOException("Unable to move " + src + " to " + dest, e);
        }
    }

    private static Path getRecoveryFile(Path corruptPath) {
        return new Path(corruptPath.getParent(), corruptPath.getName() + ".recovered");
    }

    private static boolean isReadable(Path corruptPath, Configuration conf, long maxLen) {
        try {
            OrcFile.createReader(corruptPath, OrcFile.readerOptions(conf).maxLength(maxLen));
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    private static int indexOf(byte[] data, byte[] pattern, int index) {
        if (data == null || data.length == 0 || pattern == null || pattern.length == 0 || index > data.length || index < 0) {
            return -1;
        }
        int j = 0;
        for (int i = index; i < data.length; ++i) {
            j = pattern[j] == data[i] ? ++j : 0;
            if (j != pattern.length) continue;
            return i - pattern.length + 1;
        }
        return -1;
    }

    private static void printData(List<String> files, Configuration conf) throws IOException, JSONException {
        for (String file : files) {
            try {
                Path path = new Path(file);
                Reader reader = FileDump.getReader(path, conf, Lists.newArrayList());
                if (reader == null) continue;
                FileDump.printJsonData(reader);
                System.out.println(SEPARATOR);
            }
            catch (Exception e) {
                System.err.println("Unable to dump data for file: " + file);
            }
        }
    }

    private static String getFormattedBloomFilters(int col, OrcProto.BloomFilterIndex[] bloomFilterIndex) {
        StringBuilder buf = new StringBuilder();
        BloomFilterIO stripeLevelBF = null;
        if (bloomFilterIndex != null && bloomFilterIndex[col] != null) {
            int idx = 0;
            buf.append("\n    Bloom filters for column ").append(col).append(":");
            for (OrcProto.BloomFilter bf : bloomFilterIndex[col].getBloomFilterList()) {
                BloomFilterIO toMerge = new BloomFilterIO(bf);
                buf.append("\n      Entry ").append(idx++).append(":").append(FileDump.getBloomFilterStats(toMerge));
                if (stripeLevelBF == null) {
                    stripeLevelBF = toMerge;
                    continue;
                }
                stripeLevelBF.merge(toMerge);
            }
            String bloomFilterStats = FileDump.getBloomFilterStats(stripeLevelBF);
            buf.append("\n      Stripe level merge:").append(bloomFilterStats);
        }
        return buf.toString();
    }

    private static String getBloomFilterStats(BloomFilterIO bf) {
        StringBuilder sb = new StringBuilder();
        int bitCount = bf.getBitSize();
        int popCount = 0;
        for (long l : bf.getBitSet()) {
            popCount += Long.bitCount(l);
        }
        int k = bf.getNumHashFunctions();
        float loadFactor = (float)popCount / (float)bitCount;
        float expectedFpp = (float)Math.pow(loadFactor, k);
        DecimalFormat df = new DecimalFormat("###.####");
        sb.append(" numHashFunctions: ").append(k);
        sb.append(" bitCount: ").append(bitCount);
        sb.append(" popCount: ").append(popCount);
        sb.append(" loadFactor: ").append(df.format(loadFactor));
        sb.append(" expectedFpp: ").append(expectedFpp);
        return sb.toString();
    }

    private static String getFormattedRowIndices(int col, OrcProto.RowIndex[] rowGroupIndex) {
        OrcProto.RowIndex index;
        StringBuilder buf = new StringBuilder();
        buf.append("    Row group indices for column ").append(col).append(":");
        if (rowGroupIndex == null || col >= rowGroupIndex.length || (index = rowGroupIndex[col]) == null) {
            buf.append(" not found\n");
            return buf.toString();
        }
        for (int entryIx = 0; entryIx < index.getEntryCount(); ++entryIx) {
            buf.append("\n      Entry ").append(entryIx).append(": ");
            OrcProto.RowIndexEntry entry = index.getEntry(entryIx);
            if (entry == null) {
                buf.append("unknown\n");
                continue;
            }
            OrcProto.ColumnStatistics colStats = entry.getStatistics();
            if (colStats == null) {
                buf.append("no stats at ");
            } else {
                ColumnStatisticsImpl cs = ColumnStatisticsImpl.deserialize(colStats);
                buf.append(((Object)cs).toString());
            }
            buf.append(" positions: ");
            for (int posIx = 0; posIx < entry.getPositionsCount(); ++posIx) {
                if (posIx != 0) {
                    buf.append(",");
                }
                buf.append(entry.getPositions(posIx));
            }
        }
        return buf.toString();
    }

    private static long getTotalPaddingSize(Reader reader) throws IOException {
        long paddedBytes = 0L;
        List<StripeInformation> stripes = reader.getStripes();
        for (int i = 1; i < stripes.size(); ++i) {
            long prevStripeOffset = stripes.get(i - 1).getOffset();
            long prevStripeLen = stripes.get(i - 1).getLength();
            paddedBytes += stripes.get(i).getOffset() - (prevStripeOffset + prevStripeLen);
        }
        return paddedBytes;
    }

    static Options createOptions() {
        Options result = new Options();
        OptionBuilder.withLongOpt((String)"data");
        OptionBuilder.withDescription((String)"Should the data be printed");
        result.addOption(OptionBuilder.create((char)'d'));
        OptionBuilder.withLongOpt((String)"timezone");
        OptionBuilder.withDescription((String)"Print writer's time zone");
        result.addOption(OptionBuilder.create((char)'t'));
        OptionBuilder.withLongOpt((String)"help");
        OptionBuilder.withDescription((String)"print help message");
        result.addOption(OptionBuilder.create((char)'h'));
        OptionBuilder.withLongOpt((String)"rowindex");
        OptionBuilder.withArgName((String)"comma separated list of column ids for which row index should be printed");
        OptionBuilder.withDescription((String)"Dump stats for column number(s)");
        OptionBuilder.hasArg();
        result.addOption(OptionBuilder.create());
        OptionBuilder.withLongOpt((String)"recover");
        OptionBuilder.withDescription((String)"recover corrupted orc files generated by streaming");
        result.addOption(OptionBuilder.create());
        OptionBuilder.withLongOpt((String)"skip-dump");
        OptionBuilder.withDescription((String)"used along with --recover to directly recover files without dumping");
        result.addOption(OptionBuilder.create());
        OptionBuilder.withLongOpt((String)"backup-path");
        OptionBuilder.withDescription((String)"specify a backup path to store the corrupted files (default: /tmp)");
        OptionBuilder.hasArg();
        result.addOption(OptionBuilder.create());
        return result;
    }

    private static void printMap(JSONWriter writer, Map<Object, Object> obj, List<OrcProto.Type> types, OrcProto.Type type) throws IOException, JSONException {
        writer.array();
        int keyType = type.getSubtypes(0);
        int valueType = type.getSubtypes(1);
        for (Map.Entry<Object, Object> item : obj.entrySet()) {
            writer.object();
            writer.key("_key");
            FileDump.printObject(writer, item.getKey(), types, keyType);
            writer.key("_value");
            FileDump.printObject(writer, item.getValue(), types, valueType);
            writer.endObject();
        }
        writer.endArray();
    }

    private static void printList(JSONWriter writer, List<Object> obj, List<OrcProto.Type> types, OrcProto.Type type) throws IOException, JSONException {
        int subtype = type.getSubtypes(0);
        writer.array();
        for (Object item : obj) {
            FileDump.printObject(writer, item, types, subtype);
        }
        writer.endArray();
    }

    private static void printUnion(JSONWriter writer, OrcUnion obj, List<OrcProto.Type> types, OrcProto.Type type) throws IOException, JSONException {
        int subtype = type.getSubtypes(obj.getTag());
        FileDump.printObject(writer, obj.getObject(), types, subtype);
    }

    static void printStruct(JSONWriter writer, OrcStruct obj, List<OrcProto.Type> types, OrcProto.Type type) throws IOException, JSONException {
        writer.object();
        List<Integer> fieldTypes = type.getSubtypesList();
        for (int i = 0; i < fieldTypes.size(); ++i) {
            writer.key(type.getFieldNames(i));
            FileDump.printObject(writer, obj.getFieldValue(i), types, fieldTypes.get(i));
        }
        writer.endObject();
    }

    static void printObject(JSONWriter writer, Object obj, List<OrcProto.Type> types, int typeId) throws IOException, JSONException {
        OrcProto.Type type = types.get(typeId);
        if (obj == null) {
            writer.value(null);
        } else {
            switch (type.getKind()) {
                case STRUCT: {
                    FileDump.printStruct(writer, (OrcStruct)obj, types, type);
                    break;
                }
                case UNION: {
                    FileDump.printUnion(writer, (OrcUnion)obj, types, type);
                    break;
                }
                case LIST: {
                    FileDump.printList(writer, (List)obj, types, type);
                    break;
                }
                case MAP: {
                    FileDump.printMap(writer, (Map)obj, types, type);
                    break;
                }
                case BYTE: {
                    writer.value((long)((ByteWritable)((Object)obj)).get());
                    break;
                }
                case SHORT: {
                    writer.value((long)((ShortWritable)obj).get());
                    break;
                }
                case INT: {
                    writer.value((long)((IntWritable)obj).get());
                    break;
                }
                case LONG: {
                    writer.value(((LongWritable)obj).get());
                    break;
                }
                case FLOAT: {
                    writer.value((double)((FloatWritable)obj).get());
                    break;
                }
                case DOUBLE: {
                    writer.value(((DoubleWritable)((Object)obj)).get());
                    break;
                }
                case BOOLEAN: {
                    writer.value(((BooleanWritable)obj).get());
                    break;
                }
                default: {
                    writer.value((Object)obj.toString());
                }
            }
        }
    }

    static void printJsonData(Reader reader) throws IOException, JSONException {
        OutputStreamWriter out = new OutputStreamWriter((OutputStream)System.out, "UTF-8");
        RecordReader rows = reader.rows(null);
        Object row = null;
        List<OrcProto.Type> types = reader.getTypes();
        while (rows.hasNext()) {
            row = rows.next(row);
            JSONWriter writer = new JSONWriter((java.io.Writer)out);
            FileDump.printObject(writer, row, types, 0);
            out.write("\n");
            out.flush();
        }
    }
}

