package br.com.alura.forum.controller;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Optional;

import javax.validation.Valid;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.util.UriComponentsBuilder;

import br.com.alura.forum.controller.dto.TopicoDto;
import br.com.alura.forum.controller.form.AtualizacaoTopicoForm;
import br.com.alura.forum.controller.form.NovoTopicoForm;
import br.com.alura.forum.model.Topico;
import br.com.alura.forum.model.Usuario;
import br.com.alura.forum.repository.CursoRepository;
import br.com.alura.forum.repository.TopicoRepository;
import br.com.alura.forum.repository.UsuarioRepository;

@RestController
@RequestMapping("/topicos")
public class TopicosController {

	@Autowired
	private TopicoRepository topicoRepository;

	@Autowired
	private CursoRepository cursoRepository;

	@Autowired
	private UsuarioRepository usuarioRepository;

	@GetMapping
	public List<TopicoDto> lista(@RequestParam(required = false) String nomeCurso) {
		List<Topico> todos;

		if (nomeCurso != null) {
			todos = topicoRepository.findByCursoNome(nomeCurso);
		} else {
			todos = topicoRepository.findAll();
		}

		return TopicoDto.converter(todos);
	}

	@PostMapping
	public ResponseEntity<TopicoDto> salvar(@RequestBody @Valid NovoTopicoForm form, UriComponentsBuilder uriBuilder)
			throws URISyntaxException {
		Usuario logado = usuarioRepository.getOne(1l);
		Topico novo = form.getTopico(cursoRepository, logado);
		topicoRepository.save(novo);

		URI uri = uriBuilder.path("/topicos/{id}").buildAndExpand(novo.getId()).toUri();
		return ResponseEntity.created(uri).body(new TopicoDto(novo));
	}

	@GetMapping("/{id}")
	public ResponseEntity<TopicoDto> detalhar(@PathVariable Long id) {
		Optional<Topico> detalhado = topicoRepository.findById(id);
		if (!detalhado.isPresent()) {
			return ResponseEntity.notFound().build();
		}

		return ResponseEntity.ok(new TopicoDto(detalhado.get()));
	}

	@PutMapping("/{id}")
	public ResponseEntity<?> atualizar(@PathVariable Long id, @RequestBody @Valid AtualizacaoTopicoForm form) {
		Optional<Topico> topico = topicoRepository.findById(id);
		if (!topico.isPresent()) {
			return ResponseEntity.notFound().build();
		}
		
		Topico editado = form.getTopico(cursoRepository, topicoRepository, id);
		topicoRepository.save(editado);
		return ResponseEntity.ok().build();
	}

	@DeleteMapping("/{id}")
	public ResponseEntity<?> excluir(@PathVariable Long id) {
		Optional<Topico> excluido = topicoRepository.findById(id);
		if (!excluido.isPresent()) {
			return ResponseEntity.notFound().build();
		}

		topicoRepository.delete(excluido.get());
		return ResponseEntity.ok().build();
	}

}
